"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = format;

var _fs = require("fs");

var _applyFixes = require("./apply-fixes");

var _utils = require("./utils");

/**
 * Formats the text with prettier and then eslint based on the given options
 * @param {String} options.filePath - the path of the file being formatted
 *  can be used in leu of `eslintConfig` (eslint will be used to find the
 *  relevant config for the file). Will also be used to load the `text` if
 *  `text` is not provided.
 * @param {String} options.text - the text (TypeScript code) to format
 * @param {String} options.tslintPath - the path to the tslint module to use.
 *   Will default to require.resolve('tslint')
 * @param {String} options.prettierPath - the path to the prettier module.
 *   Will default to require.resovlve('prettier')
 * @param {Object} options.tslintConfig - the config to use for formatting
 *  with TSLint.
 * @param {Object} options.prettierOptions - the options to pass for
 *  formatting with `prettier`. If not provided, prettier-eslint will attempt
 *  to create the options based on the eslintConfig
 * @param {Object} options.fallbackPrettierOptions - the options to pass for
 *  formatting with `prettier` if the given option is not inferrable from the
 *  eslintConfig.
 * @param {Boolean} options.prettierLast - Run Prettier Last
 * @return {String} - the formatted string
 */
function format(options) {
  var filePath = options.filePath,
      _options$text = options.text,
      text = _options$text === undefined ? (0, _fs.readFileSync)(filePath, "utf8") : _options$text,
      _options$tslintPath = options.tslintPath,
      tslintPath = _options$tslintPath === undefined ? (0, _utils.getModulePath)(filePath, "tslint") : _options$tslintPath,
      _options$prettierPath = options.prettierPath,
      prettierPath = _options$prettierPath === undefined ? (0, _utils.getModulePath)(filePath, "prettier") : _options$prettierPath,
      prettierLast = options.prettierLast,
      fallbackPrettierOptions = options.fallbackPrettierOptions;


  var tslintConfig = Object.assign({}, options.tslintConfig, getTSLintConfig(filePath, tslintPath));

  var prettierOptions = Object.assign({}, filePath && { filepath: filePath }, (0, _utils.getPrettierConfig)(filePath), options.prettierOptions);

  var prettify = createPrettify(prettierOptions || fallbackPrettierOptions || {}, prettierPath);
  var tslintFix = createTSLintFix(tslintConfig, tslintPath);

  if (prettierLast) {
    return prettify(tslintFix(text, filePath));
  }
  return tslintFix(prettify(text), filePath);
}

function createPrettify(formatOptions, prettierPath) {
  return function prettify(text) {
    var prettier = (0, _utils.requireModule)(prettierPath);
    try {
      var output = prettier.format(text, formatOptions);
      return output;
    } catch (error) {
      throw error;
    }
  };
}

function createTSLintFix(tslintConfig, tslintPath) {
  return function tslintFix(text, filePath) {
    var tslint = (0, _utils.requireModule)(tslintPath);
    try {
      var linter = new tslint.Linter({
        fix: false,
        formatter: "json"
      });

      linter.lint(filePath, text, tslintConfig);
      return (0, _applyFixes.applyTSLintAllFixes)(linter.getResult(), text, tslint);
    } catch (error) {
      throw error;
    }
  };
}

function getTSLintConfig(filePath, tslintPath) {
  var tslint = (0, _utils.requireModule)(tslintPath);
  try {
    return tslint.Configuration.findConfiguration(null, filePath).results;
  } catch (error) {
    return { rules: {} };
  }
}